(function(){
  // ----- Idempotency: abort previous run if this file loads again -----
  const NS = '__lsPageScripts';
  if (window[NS]?.controller) {
    try { window[NS].controller.abort(); } catch(_) {}
  }
  const controller = new AbortController();
  const signal = controller.signal;
  window[NS] = window[NS] || {};
  window[NS].controller = controller;

  // ----- Swiper (kept as-is, with null-safety) -----
  try {
    if (typeof Swiper !== 'undefined') {
      const ideasSl = document.querySelector('.ls-ideas .swiper');
      if (ideasSl) {
        new Swiper(ideasSl, {
          loop: false,
          watchSlidesProgress: false,
          centeredSlides: false,
          slidesPerView: 1.2,
          spaceBetween: 16,
          navigation: { prevEl: '.swiper-button-prev', nextEl: '.swiper-button-next' },
          pagination: { el: '.swiper-pagination' },
          breakpoints: {
            768:  { slidesPerView: 1.5, spaceBetween: 16, loop: false },
            1024: { slidesPerView: 1.5, spaceBetween: 32, loop: false },
            1280: { slidesPerView: 2, spaceBetween: 24, initialSlide: 2, centeredSlides: false, loop: true }
          }
        });
      }

      const sliders = document.querySelectorAll('.sliders .swiper');
      if (sliders && sliders.length) {
        sliders.forEach(el => {
          const count    = el.dataset.count;
          const countMob = el.dataset.countMob;
          const str      = el.dataset.loop === 'true';
          new Swiper(el, {
            speed: 800,
            slidesPerView: countMob,
            spaceBetween: 0,
            loop: str,
            allowTouchMove: true,
            breakpoints: {
              1280: { slidesPerView: count, spaceBetween: 0, allowTouchMove: false }
            }
          });
        });
      }
    }
  } catch (e) {
    console.warn('Swiper init failed:', e);
  }

  // ----- Smooth scroll for anchor links (skip clicks inside accordion) -----
  const linksHash = document.querySelectorAll('a[href^="#"]');
  linksHash.forEach(item => {
    item.addEventListener('click', function(e){
      // Let accordion handle its own toggles / buttons
      if (this.closest('.ls-accordion')) return;

      const href = this.getAttribute('href').substring(1);
      const scrollTarget = href ? document.getElementById(href) : null;
      if (scrollTarget) {
        e.preventDefault();
        document.body.classList.remove('lock');
        const elementPosition = scrollTarget.getBoundingClientRect().top;
        window.scrollBy({ top: elementPosition, behavior: 'smooth' });
      }
    }, { signal });
  });

  // ----- Buttons with data-target (null-safe for .ls-form) -----
  const buttonsHash = document.querySelectorAll('button[data-target^="#"]');
  if (buttonsHash && buttonsHash.length) {
    buttonsHash.forEach(item => {
      item.addEventListener('click', function(e){
        const id = (this.getAttribute('data-target') || '').substring(1);
        const scrollTarget = id ? document.getElementById(id) : null;
        if (scrollTarget) {
          e.preventDefault();
          const form = document.querySelector('.ls-form');
          if (form) form.classList.remove('invisible');
          document.body.classList.remove('lock');
          const elementPosition = scrollTarget.getBoundingClientRect().top;
          window.scrollBy({ top: elementPosition, behavior: 'smooth' });
        }
      }, { signal });
    });
  }

  // ----- Reveal form when near #contacts (null-safe) -----
  window.addEventListener('scroll', function(){
    const targetElement = document.getElementById('contacts');
    if (!targetElement) return;
    const scrollPosition = window.scrollY || window.pageYOffset || 0;
    if (scrollPosition >= (targetElement.offsetTop - 1500)) {
      const form = document.querySelector('.ls-form');
      if (form) form.classList.remove('invisible');
    }
  }, { signal, passive: true });

  // ----- Helpers -----
  function getSiblings(el){
    const siblings = [];
    if (!el || !el.parentNode) return siblings;
    let s = el.parentNode.firstElementChild;
    while (s) { if (s !== el) siblings.push(s); s = s.nextElementSibling; }
    return siblings;
  }

  // ----- Accordion (delegated, robust, and idempotent) -----
  // Ensure one style tag only
  if (!document.getElementById('ls-accordion-style')) {
    const style = document.createElement('style');
    style.id = 'ls-accordion-style';
    style.textContent = '.ls-accordion__wrap{overflow:hidden;max-height:0;transition:max-height .3s ease}';
    document.head.appendChild(style);
  }

  // Remove any legacy per-item listeners by NOT re-adding them; use delegation instead:
  function onDocClick(e){
    const title = e.target.closest('.ls-accordion .ls-accordion__title');
    if (!title) return;

    // If a real link inside the title was clicked, let it act normally
    if (e.target.closest('a[href]')) return;

    e.preventDefault();

    // Supports structures like: .ls-faq__item/.card-faq/.ls-accordion__item > .ls-accordion__title + .ls-accordion__wrap
    const item = title.closest('.ls-faq__item, .card-faq, .ls-accordion__item') || title.parentElement;
    const body = title.nextElementSibling;
    if (!item || !body || !body.classList.contains('ls-accordion__wrap')) return;

    // Close siblings
    const siblings = getSiblings(item);
    siblings.forEach(sib => {
      const st = sib.querySelector('.ls-accordion__title');
      const sb = sib.querySelector('.ls-accordion__wrap');
      sib.classList.remove('active');
      if (st) st.classList.remove('active');
      if (sb) sb.style.maxHeight = null;
    });

    // Toggle current
    const opening = !item.classList.contains('active');
    item.classList.toggle('active', opening);
    title.classList.toggle('active', opening);
    body.style.maxHeight = opening ? (body.scrollHeight + 'px') : null;
  }
  document.addEventListener('click', onDocClick, { signal });

  // Keep height correct if content inside expands/collapses responsively
  if (window[NS].ro) { try { window[NS].ro.disconnect(); } catch(_) {} }
  if ('ResizeObserver' in window) {
    const ro = new ResizeObserver(entries => {
      entries.forEach(entry => {
        const body = entry.target.closest('.ls-accordion__wrap');
        if (body && body.style.maxHeight) {
          body.style.maxHeight = body.scrollHeight + 'px';
        }
      });
    });
    window[NS].ro = ro;
    // Observe the body after it opens
    document.addEventListener('click', function(ev){
      const maybeBody = ev.target.closest('.ls-accordion__wrap');
      if (maybeBody) ro.observe(maybeBody);
    }, { signal });
  }

  // ----- Inputs / Textareas active state (idempotent via signal) -----
  const inputs = document.querySelectorAll('.wpsc-create-ticket input');
  function getInputValue(){
    if (inputs && inputs.length) {
      inputs.forEach(input => {
        const wrap = input.closest('.wpsc-tff');
        if (!wrap) return;
        if (input.value.trim() !== '') wrap.classList.add('active');
        else wrap.classList.remove('active');
      });
    }
  }
  getInputValue();
  inputs.forEach(el => el.addEventListener('input', getInputValue, { signal }));

  const textareas = document.querySelectorAll('.wpsc_textarea');
  function getTextareaValue(){
    if (textareas && textareas.length) {
      textareas.forEach(el => {
        const desc = el.closest('.description');
        if (!desc) return;
        if (el.value.trim() !== '') desc.classList.add('active');
        else desc.classList.remove('active');
      });
    }
  }
  getTextareaValue();
  textareas.forEach(el => el.addEventListener('input', getTextareaValue, { signal }));
})();

// SupportCandy form (safe against double-binding + double-submit)
(() => {
  const form = document.querySelector("#sc-form");
  if (!form) return;

  if (form.dataset.scBound === "1") return;
  form.dataset.scBound = "1";

  const SUPPORTCANDY_BASIC_TOKEN =
    "em9oYWlibWFsaWtpZUBnbWFpbC5jb206MHJZR2FyWlVKdWo4Z2Q0UDFDZzlxUXhh";

  const getHostUrl = () => window.location.origin ?? "https://ledstore.fi";
  const PHONE_CUSTOM_FIELD_ID = 34;

  const btn = form.querySelector('button[type="submit"]');

  const setSubmitting = (isSubmitting) => {
    if (!btn) return;
    btn.disabled = isSubmitting;
    btn.dataset.originalText ??=
      btn.querySelector("span")?.textContent || btn.textContent;

    const label = btn.querySelector("span");
    if (label) label.textContent = isSubmitting ? "Sending..." : btn.dataset.originalText;
  };

  let submitting = false;

  form.addEventListener(
    "submit",
    async (e) => {
      e.preventDefault();
      e.stopPropagation();

      if (submitting) return;
      submitting = true;

      const name = form.elements["df_customer_name"]?.value?.trim();
      const email = form.elements["df_customer_email"]?.value?.trim();
      const phone = form.elements[`cust_${PHONE_CUSTOM_FIELD_ID}`]?.value?.trim();
      const category = form.elements["category"]?.value?.trim();
      const agree = form.elements["agree"]?.checked;

      if (!name || !email || !phone) {
        alert("Please fill name, email and phone.");
        submitting = false;
        return;
      }
      if (!agree) {
        alert("Please accept the privacy policy.");
        submitting = false;
        return;
      }

      const payload = {
        is_editor: 0,
        subject: "Pikayhteys",
        name,
        email,
        category,
        [`cust_${PHONE_CUSTOM_FIELD_ID}`]: phone,
        priority: "1"
      };

      try {
        setSubmitting(true);

        const res = await fetch(`${getHostUrl()}/blog/wp-json/supportcandy/v2/tickets`, {
          method: "POST",
          credentials: "omit",
          headers: {
            Authorization: `Basic ${SUPPORTCANDY_BASIC_TOKEN}`,
            "Content-Type": "application/json",
          },
          body: JSON.stringify(payload),
        });

        const data = await res.json().catch(() => ({}));

        if (!res.ok) {
          alert(data?.message || "Submission failed.");
          return;
        }

        alert("Thanks! We received your message.");
        form.reset();
      } catch (err) {
        console.error(err);
        alert("Network error. Please try again.");
      } finally {
        setSubmitting(false);
        submitting = false;
      }
    },
    true 
  );
})();